/* Minimal SPA for EinsatzBoard */

const state = {
  user: null,
  csrf: null,
  settings: null,
  incidents: [],
  incidentId: null,
  sse: null,
  lastEventId: localStorage.getItem('lastEventId') || '0',
  page: 'dashboard',
};

const el = (id) => document.getElementById(id);
const toasts = el('toasts');

// Very small modal helper (no dependencies)
function modal({title, html, okText='OK', cancelText='Abbrechen', onOk}){
  const wrap = document.createElement('div');
  wrap.style.position='fixed';
  wrap.style.inset='0';
  wrap.style.background='rgba(0,0,0,.45)';
  wrap.style.display='flex';
  wrap.style.alignItems='center';
  wrap.style.justifyContent='center';
  wrap.style.zIndex='9999';
  wrap.innerHTML = `
    <div class="card" style="width:min(920px,96vw);max-height:88vh;overflow:auto">
      <h3>${escapeHtml(title||'')}</h3>
      <div class="body">${html||''}
        <hr />
        <div class="row" style="justify-content:flex-end">
          <button class="btn secondary" data-a="cancel">${escapeHtml(cancelText)}</button>
          <button class="btn" data-a="ok">${escapeHtml(okText)}</button>
        </div>
      </div>
    </div>`;
  document.body.appendChild(wrap);
  const close = ()=>wrap.remove();
  wrap.addEventListener('click', (e)=>{ if (e.target===wrap) close(); });
  wrap.querySelector('[data-a="cancel"]').onclick = close;
  wrap.querySelector('[data-a="ok"]').onclick = async ()=>{
    try{ await onOk?.(); } catch(e){ toast('Fehler', e.message); return; }
    close();
  };
}
async function fetchAllActiveOrgsForAdmin(){
  // Admin/EL may fetch all orgs; GF might be forbidden. We degrade gracefully.
  try{
    const r = await api('/api/admin/orgs');
    return (r.orgs||[]).filter(o=>o.is_active!==false);
  } catch(e){
    return null;
  }
}

function getDefaultOrgSelection(orgs){
  // default: user's org checked; if not found, first org.
  const ids = new Set();
  if (state.user?.org_id) ids.add(state.user.org_id);
  if (ids.size===0 && orgs?.[0]?.id) ids.add(orgs[0].id);
  return ids;
}

async function openNewIncidentModal(){
  const titleId = 'ni_title';
  const locId = 'ni_loc';
  const orgWrapId = 'ni_orgs';
  const modeInfo = (state.user?.role==='admin' || state.user?.role==='el') ? '' : '<div class="small">Hinweis: Nur Admin/EL können mehrere Organisationen auswählen.</div>';

  let orgs = await fetchAllActiveOrgsForAdmin();
  if (!orgs){
    // fallback to single org
    orgs = [{id: state.user?.org_id, name: state.user?.org_name || 'Eigene Organisation'}].filter(o=>o.id);
  }
  const defaults = getDefaultOrgSelection(orgs);

  const orgHtml = orgs.map(o=>{
    const checked = defaults.has(o.id) ? 'checked' : '';
    return `<label class="chk"><input type="checkbox" class="ni_org" value="${escapeHtml(o.id)}" ${checked}> <span>${escapeHtml(o.name||o.id)}</span></label>`;
  }).join('');

  modal({
    title:'Neuer Einsatz',
    okText:'Erstellen',
    html: `
      <div class="field"><label>Titel</label><input id="${titleId}" placeholder="z.B. Personensuche, Flächensuche..." /></div>
      <div class="field"><label>Ort</label><input id="${locId}" placeholder="optional" /></div>
      <div class="field"><label>Beteiligte Organisationen</label>
        ${modeInfo}
        <div id="${orgWrapId}" class="org-checklist">${orgHtml}</div>
        <div class="small">Mindestens 1 Organisation muss ausgewählt sein.</div>
      </div>
    `,
    onOk: async ()=>{
      const title = (document.getElementById(titleId)?.value||'').trim();
      if (!title){ toast('Fehler','Titel fehlt'); return false; }
      const loc = (document.getElementById(locId)?.value||'').trim();
      const orgIds = Array.from(document.querySelectorAll('#'+orgWrapId+' .ni_org:checked')).map(x=>x.value);
      if (!orgIds.length){ toast('Fehler','Bitte mindestens 1 Organisation auswählen'); return false; }
      await api('/api/incidents', {method:'POST', body:{title, location_text: loc, org_ids: orgIds}});
      await loadIncidents();
      toast('OK','Einsatz erstellt');
      return true;
    }
  });
}


function toast(title, body, actions=[]) {
  const d = document.createElement('div');
  d.className='toast';
  d.innerHTML = `<div class="t">${escapeHtml(title)}</div><div class="small">${escapeHtml(body||'')}</div>`;
  if (actions.length) {
    const a = document.createElement('div');
    a.className='a';
    for (const act of actions) {
      const b = document.createElement('button');
      b.className = 'btn secondary';
      b.textContent = act.label;
      b.onclick = () => { try{act.onClick();}finally{d.remove();} };
      a.appendChild(b);
    }
    d.appendChild(a);
  }
  toasts.appendChild(d);
  setTimeout(()=>d.remove(), 12000);
}

function escapeHtml(s){
  return String(s).replace(/[&<>"']/g, c => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[c]));
}

async function api(path, {method='GET', body, isForm=false}={}) {
  const headers = {};
  if (!isForm) headers['Content-Type']='application/json';
  if (state.csrf) headers['X-CSRF-Token']=state.csrf;
  const res = await fetch(path, {
    method,
    headers,
    body: body ? (isForm ? body : JSON.stringify(body)) : undefined,
    credentials: 'same-origin',
  });
  const ct = res.headers.get('content-type')||'';
  const data = ct.includes('application/json') ? await res.json() : {ok:false, error: await res.text()};
  if (!res.ok || data.ok===false) {
    throw new Error(data.error || `HTTP ${res.status}`);
  }
  return data;
}

function applyBrand(settings){
  const b = settings?.branding;
  if (!b) return;
  document.documentElement.style.setProperty('--primary', b.primaryColor || '#d6002a');
  document.documentElement.style.setProperty('--accent', b.accentColor || '#111827');
  el('brandName').textContent = b.appName || 'EinsatzBoard';
  document.title = b.appName || 'EinsatzBoard';
  const logo = el('brandLogo');
  if (b.logoPath) { logo.src=b.logoPath; logo.style.display='block'; } else { logo.style.display='none'; }
  if (b.faviconPath) {
    let link = document.querySelector('link[rel="icon"]');
    if (!link) { link=document.createElement('link'); link.rel='icon'; document.head.appendChild(link); }
    link.href=b.faviconPath;
  }
}

function setLoggedInUI(on){
  el('viewLogin').style.display = on ? 'none' : 'block';
  el('viewApp').style.display = on ? 'block' : 'none';
  el('btnLogout').style.display = on ? 'inline-block' : 'none';
}

async function boot(){
  try{
    const me = await api('/api/me');
    state.user = me.user;
    state.csrf = me.csrf;
    state.settings = me.settings;
    applyBrand(state.settings);

    if (!state.user) {
      setLoggedInUI(false);
      const pub = await api('/api/public-settings');
      applyBrand(pub.settings);
      el('loginHint').textContent = 'Default seed: admin / admin123 (bitte danach ändern)';
      return;
    }

    el('brandOrg').textContent = `${state.user.display_name} · ${state.user.role}`;
    setLoggedInUI(true);
    await loadIncidents();
    buildNav();
    connectSSE();
    render();
  } catch(e){
    console.error(e);
    toast('Fehler', e.message);
    setLoggedInUI(false);
  }
}

async function loadIncidents(){
  const res = await api('/api/incidents');
  state.incidents = res.incidents;
  const sel = el('incidentSelect');
  sel.innerHTML='';
  const opt0 = document.createElement('option');
  opt0.value='';
  opt0.textContent = '— Einsatz wählen —';
  sel.appendChild(opt0);
  for (const i of state.incidents) {
    const o = document.createElement('option');
    o.value=i.id;
    o.textContent = `${i.title} · ${i.status}`;
    sel.appendChild(o);
  }
  const saved = localStorage.getItem('incidentId');
  if (saved && state.incidents.some(i=>i.id===saved)) state.incidentId=saved;
  sel.value = state.incidentId || '';
}

function buildNav(){
  const pages = [
    ['dashboard','Dashboard'],
    ['participants','Teilnehmer'],
    ['teams','Teams'],
    ['orders','Aufträge'],
    ['inbox','Inbox'],
    ['chat','Chat'],
  ];
  if (state.user?.role==='admin') pages.push(['admin','Admin']);

  const nav = el('nav');
  nav.innerHTML='';
  for (const [key,label] of pages) {
    const b = document.createElement('button');
    b.textContent = label;
    b.className = state.page===key ? 'active' : '';
    b.onclick = ()=>{ state.page=key; buildNav(); render(); };
    nav.appendChild(b);
  }
}

function requireIncident(){
  if (!state.incidentId) throw new Error('Bitte zuerst einen Einsatz auswählen');
}

async function render(){
  const titleMap = {
    dashboard:'Dashboard', participants:'Teilnehmer', teams:'Teams', orders:'Aufträge', inbox:'Inbox', chat:'Chat', admin:'Admin Settings'
  };
  el('pageTitle').textContent = titleMap[state.page] || '...';

  const page = el('page');
  page.innerHTML='';

  if (state.page==='dashboard') return renderDashboard(page);
  if (state.page==='participants') return renderParticipants(page);
  if (state.page==='teams') return renderTeams(page);
  if (state.page==='orders') return renderOrders(page);
  if (state.page==='inbox') return renderInbox(page);
  if (state.page==='chat') return renderChat(page);
  if (state.page==='admin') return renderAdmin(page);
}

async function renderDashboard(root){
  root.innerHTML = `
    <div class="row">
      <button class="btn" id="btnNewIncident">Neuer Einsatz</button>
      <button class="btn secondary" id="btnLoad">Daten neu laden</button>
    </div>
    <hr />
    <div class="small">Live-Updates: aktiv (SSE). Letztes Event: ${escapeHtml(String(state.lastEventId))}</div>
  `;

  el('btnLoad').onclick = async ()=>{ await refreshAll(); };
  el('btnNewIncident').onclick = async ()=>{
    await openNewIncidentModal();
  };
}

async function renderParticipants(root){
  try{ requireIncident(); } catch(e){ root.innerHTML = `<div class="small">${escapeHtml(e.message)}</div>`; return; }

  // Phase B: roster-driven view + directory editor
  state._partsTab = state._partsTab || 'checkin';
  const [rosterRes, partsRes, capsRes, dirRes] = await Promise.all([
    api(`/api/incidents/${state.incidentId}/roster`),
    api(`/api/incidents/${state.incidentId}/participants`),
    api(`/api/capabilities`),
    api(`/api/members`),
  ]);
  const roster = rosterRes.roster;
  const parts = partsRes.participants;
  const members = dirRes.members.filter(m=>m.is_active);
  const caps = capsRes.capabilities || [];
  const capKeys = caps.map(c=>c.key);

  root.innerHTML = `
    <div class="row" style="gap:8px">
      <button class="btn ${state._partsTab==='checkin'?'':'secondary'}" id="tabCheckin">Check-In/Out</button>
      <button class="btn ${state._partsTab==='directory'?'':'secondary'}" id="tabDir">Mitgliederverzeichnis</button>
    </div>
    <hr />
    <div id="partsBody"></div>
    <datalist id="capList">${capKeys.map(k=>`<option value="${escapeHtml(k)}"></option>`).join('')}</datalist>
  `;

  el('tabCheckin').onclick = ()=>{ state._partsTab='checkin'; render(); };
  el('tabDir').onclick = ()=>{ state._partsTab='directory'; render(); };

  const body = el('partsBody');
  if (state._partsTab==='directory') {
    body.innerHTML = renderMemberDirectoryHtml();
    wireMemberDirectory(members);
    return;
  }

  body.innerHTML = `
    <div class="row" style="align-items:flex-start">
      <div style="min-width:360px;flex:1">
        <div class="small">Schnell-Suche</div>
        <input id="qRoster" placeholder="Name / Code / Fähigkeit" style="width:100%" />
        <div class="row" style="margin-top:8px">
          <button class="btn" id="btnQuick">Schnell Check-In (neu)</button>
          <button class="btn secondary" id="btnCheckinSel">Auswahl Check-In</button>
        </div>
        <div class="small" style="margin-top:10px">Auswahl</div>
        <select id="selRoster" multiple size="10" style="width:100%"></select>
        <div class="row" style="margin-top:8px">
          <button class="btn danger" id="btnCheckoutAll">Alle Check-Out</button>
        </div>
      </div>
      <div style="flex:2">
        <div class="row" style="justify-content:space-between;align-items:center">
          <div class="small">Teilnehmer (eingecheckt): ${parts.length}</div>
          <label class="small"><input type="checkbox" id="onlyIn" checked /> nur eingecheckt</label>
        </div>
        <div class="list" id="participantList"></div>
      </div>
    </div>
  `;

  // fill roster select
  const sel = el('selRoster');
  const fill = (q='')=>{
    sel.innerHTML='';
    const qq = q.trim().toLowerCase();
    for (const r of roster) {
      const hay = `${r.display_name||''} ${r.short_code||''} ${(r.capabilities||[]).join(' ')}`.toLowerCase();
      if (qq && !hay.includes(qq)) continue;
      const o = document.createElement('option');
      o.value = r.member_id;
      const loanTag = r.loaned_in ? ' (geliehen)' : '';
      o.textContent = `${r.display_name}${r.short_code?` (${r.short_code})`:''}${loanTag} · ${r.status}`;
      sel.appendChild(o);
    }
  };
  fill('');
  el('qRoster').oninput = (e)=>fill(e.target.value);

  el('btnCheckinSel').onclick = async ()=>{
    const ids = Array.from(sel.selectedOptions).map(o=>o.value);
    if (!ids.length) return toast('Hinweis','Bitte Mitglieder auswählen');
    await api(`/api/incidents/${state.incidentId}/checkin`, {method:'POST', body:{member_ids:ids}});
    toast('OK','Eingecheckt');
    await render();
  };

  el('btnQuick').onclick = async ()=>{
    const display_name = prompt('Name (neu anlegen + Check-In):');
    if (!display_name) return;
    const short_code = prompt('Kurzcode (optional):') || '';
    const capStr = prompt(`Fähigkeiten (keys, kommasepariert)\nBekannt: ${capKeys.join(', ')}`) || '';
    const capabilities = capStr.split(',').map(s=>s.trim()).filter(Boolean);
    await api(`/api/incidents/${state.incidentId}/checkin-quick`, {method:'POST', body:{display_name, short_code, capabilities}});
    toast('OK','Neu angelegt & eingecheckt');
    await render();
  };

  el('btnCheckoutAll').onclick = async ()=>{
    const reason = prompt('Checkout-Grund (optional):')||'';
    await api(`/api/incidents/${state.incidentId}/checkout`, {method:'POST', body:{all:true, reason}});
    toast('OK','Alle ausgecheckt');
    await render();
  };

  const list = el('participantList');
  const renderList = ()=>{
    const onlyIn = el('onlyIn').checked;
    list.innerHTML='';
    for (const p of parts) {
      if (onlyIn && p.status!=='in') continue;
      const it = document.createElement('div');
      it.className='item';
      const name = p.display_name || p.pseudonym || '(anonym)';
      const loan = p.loan ? ` · Leihe: ${String(p.loan.from_org_id).slice(0,8)} → ${String(p.loan.to_org_id).slice(0,8)}` : '';
      it.innerHTML = `
        <div class="row" style="justify-content:space-between">
          <div>
            <div class="title">${escapeHtml(name)} <span class="small">${escapeHtml(p.org_short||'')}</span></div>
            <div class="small">Status: ${escapeHtml(p.status)} · In: ${escapeHtml(p.checked_in_at||'')} ${p.checked_out_at?`· Out: ${escapeHtml(p.checked_out_at)}`:''}${escapeHtml(loan)}</div>
            <div class="small">Fähigkeiten: ${(p.capabilities||[]).map(escapeHtml).join(', ')||'—'}</div>
          </div>
          <div class="row">
            <button class="btn secondary" data-mid="${p.member_id}" ${p.status==='out'?'disabled':''}>Einzeln Check-Out</button>
          </div>
        </div>
      `;
      it.querySelector('button').onclick = async ()=>{
        const r = prompt('Grund (optional):')||'';
        await api(`/api/incidents/${state.incidentId}/checkout`, {method:'POST', body:{member_id:p.member_id, reason:r}});
        await render();
      };
      list.appendChild(it);
    }
  };
  renderList();
  el('onlyIn').onchange = renderList;
}

function renderMemberDirectoryHtml(){
  return `
    <div class="row" style="justify-content:space-between;align-items:center">
      <div class="small">Mitglieder deiner Organisation</div>
      <button class="btn" id="btnAddMember">Neues Mitglied</button>
    </div>
    <div class="small" style="margin-top:8px">Suche</div>
    <input id="qMembers" placeholder="Name / Kurzcode / Fähigkeit" style="width:100%" />
    <hr />
    <div class="list" id="memberList"></div>
  `;
}

function wireMemberDirectory(members){
  const list = el('memberList');
  const renderList = (q='')=>{
    const qq = q.trim().toLowerCase();
    list.innerHTML='';
    for (const m of members) {
      const hay = `${m.display_name||''} ${m.short_code||''} ${(m.capabilities||[]).join(' ')}`.toLowerCase();
      if (qq && !hay.includes(qq)) continue;
      const it = document.createElement('div');
      it.className='item';
      it.innerHTML = `
        <div class="row" style="justify-content:space-between">
          <div>
            <div class="title">${escapeHtml(m.display_name)} <span class="small">${escapeHtml(m.short_code||'')}</span></div>
            <div class="small">Fähigkeiten: ${(m.capabilities||[]).map(escapeHtml).join(', ')||'—'}</div>
          </div>
          <div class="row">
            <button class="btn secondary">Bearbeiten</button>
          </div>
        </div>
      `;
      it.querySelector('button').onclick = ()=>openMemberEditor(m);
      list.appendChild(it);
    }
    if (!list.children.length) list.innerHTML = `<div class="small">Keine Treffer.</div>`;
  };
  renderList('');
  el('qMembers').oninput = (e)=>renderList(e.target.value);
  el('btnAddMember').onclick = ()=>openMemberEditor(null);
}

function openMemberEditor(member){
  const name = prompt('Name:', member?.display_name || '');
  if (name===null) return;
  if (!name.trim()) return toast('Fehler','Name erforderlich');
  const short_code = prompt('Kurzcode (optional):', member?.short_code || '') || '';
  const capStr = prompt('Fähigkeiten (keys, kommasepariert):', (member?.capabilities||[]).join(', ')) || '';
  const capabilities = capStr.split(',').map(s=>s.trim()).filter(Boolean);
  api('/api/members', {method:'POST', body:{id: member?.id, display_name:name.trim(), short_code:short_code.trim(), capabilities}})
    .then(()=>{ toast('OK','Gespeichert'); render(); })
    .catch(e=>toast('Fehler', e.message));
}

async function renderTeams(root){
  try{ requireIncident(); } catch(e){ root.innerHTML = `<div class="small">${escapeHtml(e.message)}</div>`; return; }

  const data = await api(`/api/incidents/${state.incidentId}/teams`);
  const teams = data.teams;
  root.innerHTML = `
    <div class="row">
      <button class="btn" id="btnNewTeam">Neues Team</button>
    </div>
    <hr />
    <div class="list" id="teamList"></div>
  `;

  el('btnNewTeam').onclick = async ()=>{
    const name = prompt('Teamname?');
    if (!name) return;
    const callsign = prompt('Rufname (optional)?')||'';
    await api(`/api/incidents/${state.incidentId}/teams`, {method:'POST', body:{name, callsign}});
    toast('OK','Team erstellt');
    await render();
  };

  const list = el('teamList');
  list.innerHTML='';
  const roleDefs = state.settings?.team?.roles || [];
  const roleLabel = (k)=> (roleDefs.find(r=>r.key===k)?.label) || k;

  for (const t of teams) {
    const it = document.createElement('div');
    it.className='item';
    it.innerHTML = `
      <div class="row" style="justify-content:space-between">
        <div>
          <div class="title">${escapeHtml(t.name)} <span class="small">${escapeHtml(t.org_short||'')}</span></div>
          <div class="small">Status: ${escapeHtml(t.status||'')} · Rufname: ${escapeHtml(t.callsign||'')}</div>
          <div class="small">Mitglieder: ${(t.members||[]).map(m=>{
            const r = roleLabel(m.role_in_team||'member');
            return `${escapeHtml(m.display_name)} (${escapeHtml(r)})`;
          }).join(', ')||'—'}</div>
        </div>
        <div class="row">
          <button class="btn secondary">Mitglieder zuweisen</button>
        </div>
      </div>
    `;
    it.querySelector('button').onclick = ()=>openAssignTeam(t);
    list.appendChild(it);
  }
}

async function openAssignTeam(team){
  requireIncident();
  const [parts, rosterRes] = await Promise.all([
    api(`/api/incidents/${state.incidentId}/participants`),
    api(`/api/incidents/${state.incidentId}/roster`)
  ]);
  const avail = parts.participants.filter(p=>p.status==='in');
  const roleDefs = state.settings?.team?.roles || [{key:'member',label:'Mitglied',sort:90}];
  const roleOpts = roleDefs
    .slice()
    .sort((a,b)=>(Number(a.sort||100)-Number(b.sort||100)))
    .map(r=>`<option value="${escapeHtml(r.key)}">${escapeHtml(r.label)}</option>`)
    .join('');

  const current = new Map();
  for (const m of (team.members||[])) {
    current.set(m.member_id, {role: m.role_in_team||'member', is_lead: !!m.is_lead});
  }

  const rowsHtml = avail.map(p=>{
    const cur = current.get(p.member_id);
    const checked = cur ? 'checked' : '';
    const role = cur?.role || 'member';
    const lead = cur?.is_lead ? 'checked' : '';
    const name = escapeHtml(p.display_name || p.pseudonym || p.member_id);
    const org = escapeHtml(p.org_short||p.org_name||'');
    return `
      <tr>
        <td><input type="checkbox" data-mid="${escapeHtml(p.member_id)}" ${checked}></td>
        <td>${name}<div class="small">${org}</div></td>
        <td><select data-role-for="${escapeHtml(p.member_id)}">${roleOpts}</select></td>
        <td style="text-align:center"><input type="checkbox" data-lead-for="${escapeHtml(p.member_id)}" ${lead}></td>
      </tr>`;
  }).join('');

  modal({
    title: `Team: ${team.name} – Mitglieder & Rollen`,
    okText: 'Speichern',
    html: `
      <div class="small">Nur eingecheckte Teilnehmer sind zuweisbar. Rollen kommen aus Admin → Einstellungen.</div>
      <table class="table" style="width:100%;margin-top:8px">
        <thead><tr><th></th><th>Mitglied</th><th>Rolle</th><th>Lead</th></tr></thead>
        <tbody>${rowsHtml || '<tr><td colspan="4" class="small">Keine eingecheckten Teilnehmer.</td></tr>'}</tbody>
      </table>
    `,
    onOk: async ()=>{
      const members = [];
      document.querySelectorAll('input[type="checkbox"][data-mid]').forEach(cb=>{
        if (!cb.checked) return;
        const mid = cb.getAttribute('data-mid');
        const sel = document.querySelector(`select[data-role-for="${CSS.escape(mid)}"]`);
        const leadCb = document.querySelector(`input[data-lead-for="${CSS.escape(mid)}"]`);
        members.push({member_id: mid, role: sel?.value || 'member', is_lead: !!leadCb?.checked});
      });
      await api(`/api/teams/${team.id}/assign`, {method:'POST', body:{members}});
      toast('OK','Team aktualisiert');
      await render();
    }
  });
}

async function renderOrders(root){
  try{ requireIncident(); } catch(e){ root.innerHTML = `<div class="small">${escapeHtml(e.message)}</div>`; return; }

  const data = await api(`/api/incidents/${state.incidentId}/orders`);
  const orders = data.orders;
  root.innerHTML = `
    <div class="row">
      <button class="btn" id="btnNewOrder">Neuer Auftrag</button>
    </div>
    <hr />
    <div class="list" id="orderList"></div>
  `;

  el('btnNewOrder').onclick = async ()=>{
    const tRes = await api(`/api/incidents/${state.incidentId}/teams`);
    const teams = tRes.teams.filter(t=>t.org_id===state.user.org_id);
    const templates = state.settings?.orderTemplates || [];

    const tplOpts = templates.map(t=>`<option value="${escapeHtml(t.key)}">${escapeHtml(t.label||t.key)}</option>`).join('');
    const teamOpts = [`<option value="">—</option>`].concat(teams.map(t=>`<option value="${escapeHtml(t.id)}">${escapeHtml(t.name)}</option>`)).join('');

    const renderFields = (tplKey)=>{
      const tpl = templates.find(t=>t.key===tplKey) || templates[0];
      const fields = Array.isArray(tpl?.fields)?tpl.fields:[];
      const out = [];
      for (const f of fields) {
        const key = f.key;
        const label = f.label||key;
        const req = f.required ? ' *' : '';
        if (f.type==='textarea') {
          out.push(`<div class="field"><label>${escapeHtml(label)}${req}</label><textarea data-f="${escapeHtml(key)}" style="width:100%;min-height:70px"></textarea></div>`);
        } else if (f.type==='select') {
          const opts = (f.options||[]).map(o=>`<option value="${escapeHtml(String(o))}">${escapeHtml(String(o))}</option>`).join('');
          out.push(`<div class="field"><label>${escapeHtml(label)}${req}</label><select data-f="${escapeHtml(key)}" style="width:100%"><option value=""></option>${opts}</select></div>`);
        } else {
          out.push(`<div class="field"><label>${escapeHtml(label)}${req}</label><input data-f="${escapeHtml(key)}" style="width:100%" /></div>`);
        }
      }
      return out.join('');
    };

    modal({
      title:'Neuer Auftrag',
      okText:'Erstellen',
      html:`
        <div class="split">
          <div>
            <div class="field"><label>Template</label><select id="o_tpl" style="width:100%">${tplOpts}</select></div>
            <div class="field"><label>Titel</label><input id="o_title" style="width:100%" placeholder="(optional, sonst Template-Name)"/></div>
            <div class="row"><div class="field" style="flex:1"><label>Priorität</label><input id="o_prio" type="number" min="1" max="5" value="3" style="width:100%"/></div>
              <div class="field" style="flex:2"><label>Team (optional)</label><select id="o_team" style="width:100%">${teamOpts}</select></div></div>
          </div>
          <div>
            <div class="small">Felder aus Template</div>
            <div id="o_fields" style="margin-top:6px">${renderFields(templates[0]?.key||'')}</div>
          </div>
        </div>
      `,
      onOk: async ()=>{
        const template_key = document.getElementById('o_tpl').value;
        const title = document.getElementById('o_title').value.trim();
        const priority = Number(document.getElementById('o_prio').value||3);
        const assigned_team_id = document.getElementById('o_team').value;
        const fields = {};
        document.querySelectorAll('#o_fields [data-f]').forEach(n=>{
          const k = n.getAttribute('data-f');
          const v = (n.value||'').trim();
          if (v!=='') fields[k]=v;
        });
        await api(`/api/incidents/${state.incidentId}/orders`, {method:'POST', body:{template_key, title, priority, assigned_team_id, fields}});
        toast('OK','Auftrag erstellt');
        await render();
      }
    });
    // wire template change
    setTimeout(()=>{
      const sel = document.getElementById('o_tpl');
      if (!sel) return;
      sel.onchange = ()=>{ document.getElementById('o_fields').innerHTML = renderFields(sel.value); };
    },0);
  };

  const list = el('orderList');
  list.innerHTML='';
  for (const o of orders) {
    const it = document.createElement('div');
    it.className='item';
    it.innerHTML = `
      <div class="row" style="justify-content:space-between;align-items:flex-start">
        <div>
          <div class="title">${escapeHtml(o.title)} <span class="small">${escapeHtml(o.status)}</span></div>
          <div class="small">Org: ${escapeHtml(o.assigned_org_name||'')} · Team: ${escapeHtml(o.team_name||'—')} · Prio: ${escapeHtml(String(o.priority))}</div>
        </div>
        <div class="row">
          <button class="btn secondary" data-a="note">Notiz</button>
          <button class="btn secondary" data-a="transfer">Übergeben</button>
          <button class="btn" data-a="next">Next Status</button>
        </div>
      </div>
    `;
    const btns = it.querySelectorAll('button');
    btns.forEach(b=>{
      b.onclick = async ()=>{
        const a = b.dataset.a;
        if (a==='note') {
          const m = prompt('Notiz:'); if (!m) return;
          await api(`/api/orders/${o.id}/note`, {method:'POST', body:{message:m}});
          return toast('OK','Notiz gespeichert');
        }
        if (a === 'transfer') {
          openOrderTransferModal(o.id);
          return;
        }
        if (a==='next') {
          const next = ({created:'accepted', accepted:'started', started:'completed', completed:'completed'})[o.status] || 'accepted';
          await api(`/api/orders/${o.id}`, {method:'PUT', body:{status:next}});
          toast('OK',`Status -> ${next}`);
          await render();
        }
      }
    });
    list.appendChild(it);
  }
}

async function openOrderTransferModal(orderId){
  const incId =
    state.incidentId ||
    state.currentIncidentId ||
    state.currentEinsatzId ||
    state.einsatzId ||
    state.incident_id ||
    state.current?.incidentId ||
    null;

  if(!incId) return alert('Kein Einsatz gewählt');

  // Hole Einsatzdetails inkl. beteiligter Orgs
  const inc = await API(`/api/incidents/${incId}`);

  // Erwartet: inc.orgs = [{id,name,short_name,color}, ...]
  const orgs = (inc.orgs || []).filter(o => o.id !== state.me?.org_id);

  if(orgs.length === 0){
    return alert('Im Einsatz sind keine weiteren Organisationen hinterlegt.');
  }

  // Default Transfer Mode aus Settings
  const defMode = state.settings?.ops?.transferModeDefault || 'offer'; // offer|fast
  const grace = state.settings?.ops?.fastModeGraceSeconds ?? 120;

  // Modal HTML (Org-Auswahl statt UUID)
  const options = orgs.map(o =>
    `<option value="${o.id}">${escapeHtml(o.short_name ? `${o.short_name} – ${o.name}` : o.name)}</option>`
  ).join('');

  modal.open({
    title: 'Auftrag übergeben',
    html: `
      <div class="field">
        <label>Ziel-Organisation</label>
        <select id="tr_to_org">${options}</select>
      </div>

      <div class="field">
        <label>Modus</label>
        <select id="tr_mode">
          <option value="offer"${defMode==='offer'?' selected':''}>Angebot (Annehmen/Ablehnen)</option>
          <option value="fast"${defMode==='fast'?' selected':''}>Fast (Auto nach ${grace}s)</option>
        </select>
      </div>

      <div class="hint">
        Fast-Modus erzeugt keinen Flaschenhals: wenn niemand reagiert, wird automatisch angenommen.
      </div>
    `,
    onOk: async ()=>{
      const to_org_id = document.getElementById('tr_to_org').value;
      const mode = document.getElementById('tr_mode').value;

      await API(`/api/orders/${orderId}/transfer`, {}, {
        method: 'POST',
        body: JSON.stringify({ to_org_id, mode })
      });

      toast('Übergabe gesendet');
    }
  });
}

async function renderInbox(root){
  const data = await api('/api/inbox');
  const rows = data.inbox;
  root.innerHTML = `<div class="list" id="inboxList"></div>`;
  const list = el('inboxList');
  list.innerHTML='';
  if (!rows.length) {
    list.innerHTML = `<div class="small">Keine offenen Angebote.</div>`;
    return;
  }
  for (const r of rows) {
    const it = document.createElement('div');
    it.className='item';
    const what = r.type==='order_transfer' ? `Auftrag: ${r.payload.order_title}` : `Mitglied: ${r.payload.member_name}`;
    it.innerHTML = `
      <div class="row" style="justify-content:space-between">
        <div>
          <div class="title">${escapeHtml(what)}</div>
          <div class="small">Von: ${escapeHtml(r.from_org_name)} · Modus: ${escapeHtml(r.mode)} ${r.expires_at?`· Ablauf: ${escapeHtml(r.expires_at)}`:''}</div>
        </div>
        <div class="row">
          <button class="btn">Annehmen</button>
          <button class="btn danger">Ablehnen</button>
        </div>
      </div>
    `;
    const [b1,b2] = it.querySelectorAll('button');
    b1.onclick = async ()=>{ await api(`/api/transfer/${r.id}/decide`, {method:'POST', body:{decision:'accept'}}); toast('OK','Angenommen'); await render(); };
    b2.onclick = async ()=>{ await api(`/api/transfer/${r.id}/decide`, {method:'POST', body:{decision:'reject'}}); toast('OK','Abgelehnt'); await render(); };
    list.appendChild(it);
  }
}

async function renderChat(root){
  try{ requireIncident(); } catch(e){ root.innerHTML = `<div class="small">${escapeHtml(e.message)}</div>`; return; }

  const data = await api(`/api/incidents/${state.incidentId}/chat`);
  const msgs = data.messages;
  root.innerHTML = `
    <div class="list" id="chatList" style="max-height:420px;overflow:auto"></div>
    <hr />
    <div class="row">
      <select id="chatScope">
        <option value="public">Öffentlich (Einsatz)</option>
        <option value="org">Eigene Org</option>
        <option value="gfs">GFs/EL</option>
      </select>
      <input id="chatMsg" placeholder="Nachricht... (@mention)" style="flex:1" />
      <button class="btn" id="btnSend">Senden</button>
    </div>
  `;

  const list = el('chatList');
  list.innerHTML='';
  for (const m of msgs) {
    const d = document.createElement('div');
    d.className='item';
    d.innerHTML = `<div class="small">${escapeHtml(m.created_at)} · ${escapeHtml(m.author_name)} · ${escapeHtml(m.scope)}</div><div>${escapeHtml(m.message)}</div>`;
    list.appendChild(d);
  }
  list.scrollTop = list.scrollHeight;

  el('btnSend').onclick = async ()=>{
    const message = el('chatMsg').value.trim();
    if (!message) return;
    const scope = el('chatScope').value;
    await api(`/api/incidents/${state.incidentId}/chat`, {method:'POST', body:{scope, message}});
    el('chatMsg').value='';
    await render();
  };
}

async function renderAdmin(root){
  const adminTab = localStorage.getItem('adminTab') || 'settings';
  const tabs = [
    ['settings','Einstellungen'],
    ['templates','Auftrags-Templates'],
    ['orgs','Organisationen'],
    ['users','Benutzer'],
  ];
  root.innerHTML = `
    <div class="row" id="adminTabs"></div>
    <hr />
    <div id="adminBody"></div>
  `;
  const tabWrap = root.querySelector('#adminTabs');
  for (const [k, label] of tabs) {
    const b = document.createElement('button');
    b.className = 'btn secondary';
    if (k === adminTab) b.className = 'btn';
    b.textContent = label;
    b.onclick = ()=>{ localStorage.setItem('adminTab', k); render(); };
    tabWrap.appendChild(b);
  }
  const body = root.querySelector('#adminBody');

  if (adminTab === 'settings') return renderAdminSettings(body);
  if (adminTab === 'templates') return renderAdminTemplates(body);
  if (adminTab === 'orgs') return renderAdminOrgs(body);
  if (adminTab === 'users') return renderAdminUsers(body);
}

async function renderAdminSettings(root){
  const rows = (await api('/api/settings')).settings;
  const map = Object.fromEntries(rows.map(r=>[r.key, r.value]));
  const branding = map.branding || {};
  const privacy = map.privacy || {};
  const ops = map.ops || {};
  const team = map.team || {};
  const roles = Array.isArray(team.roles) ? team.roles : [];
  const rolesText = roles
    .sort((a,b)=>(Number(a.sort||100)-Number(b.sort||100)))
    .map(r=>`${r.key||''}|${r.label||''}|${r.sort??''}`)
    .join('\n');

  root.innerHTML = `
    <div class="split">
      <div>
        <div class="small">Branding</div>
        <div class="row"><input id="s_appName" placeholder="Tool-Name" style="flex:1" value="${escapeHtml(branding.appName||'')}" /></div>
        <div class="row"><input id="s_primary" placeholder="#Primary" value="${escapeHtml(branding.primaryColor||'')}" />
          <input id="s_accent" placeholder="#Accent" value="${escapeHtml(branding.accentColor||'')}" /></div>
        <div class="row"><button class="btn secondary" id="btnLogo">Logo upload</button><button class="btn secondary" id="btnFav">Favicon upload</button></div>
        <div class="small">Logo: ${escapeHtml(branding.logoPath||'—')}</div>
        <div class="small">Favicon: ${escapeHtml(branding.faviconPath||'—')}</div>
      </div>
      <div>
        <div class="small">Datenschutz</div>
        <label class="small"><input type="checkbox" id="s_pseudo" ${privacy.pseudonymizeOtherOrgs?'checked':''}/> Andere Orgs pseudonymisieren</label><br/>
        <label class="small"><input type="checkbox" id="s_showNames" ${privacy.showNamesToOtherOrgs?'checked':''}/> Klarnamen für andere Orgs anzeigen</label><br/>
        <div class="row"><input id="s_retention" type="number" min="0" value="${escapeHtml(String(privacy.retentionDays??180))}" /> <span class="small">Retention (Tage)</span></div>
        <hr />
        <div class="small">Betrieb</div>
        <div class="row"><select id="s_transferMode"><option value="offer">offer</option><option value="fast">fast</option></select>
          <input id="s_fastGrace" type="number" min="0" value="${escapeHtml(String(ops.fastModeGraceSeconds??120))}" /> <span class="small">Fast-Grace (sek)</span></div>
        <div class="row">
          <input id="s_hbWarn" type="number" min="0" value="${escapeHtml(String(ops.heartbeatWarnMinutes??30))}" /> <span class="small">Heartbeat Warn (min)</span>
        </div>
        <div class="row">
          <input id="s_hbAlarm" type="number" min="0" value="${escapeHtml(String(ops.heartbeatAlarmMinutes??45))}" /> <span class="small">Heartbeat Alarm (min)</span>
        </div>

        <hr />
        <div class="small">Team-Rollen (frei definierbar)</div>
        <div class="small">Format pro Zeile: <code>key|Label|Sort</code> (Sort optional)</div>
        <textarea id="s_teamRoles" style="width:100%;min-height:120px" placeholder="lead|Leitung|10\nnav|Navigation|20\nradio|Funk|30\nmember|Mitglied|90">${escapeHtml(rolesText)}</textarea>
        <label class="small"><input type="checkbox" id="s_multiLead" ${team.allowMultipleLeads!==false?'checked':''}/> Mehrere Leitungs-Rollen pro Team erlauben</label>
      </div>
    </div>
    <hr />
    <button class="btn" id="btnSaveSettings">Speichern</button>
  `;
  el('s_transferMode').value = ops.transferModeDefault || 'offer';

  el('btnSaveSettings').onclick = async ()=>{
    const newBrand = {
      appName: el('s_appName').value.trim(),
      primaryColor: el('s_primary').value.trim(),
      accentColor: el('s_accent').value.trim(),
      logoPath: branding.logoPath || null,
      faviconPath: branding.faviconPath || null,
    };
    const newPrivacy = {
      pseudonymizeOtherOrgs: el('s_pseudo').checked,
      showNamesToOtherOrgs: el('s_showNames').checked,
      retentionDays: Number(el('s_retention').value||0),
    };
    const newOps = {
      transferModeDefault: el('s_transferMode').value,
      fastModeGraceSeconds: Number(el('s_fastGrace').value||0),
      heartbeatWarnMinutes: Number(el('s_hbWarn').value||0),
      heartbeatAlarmMinutes: Number(el('s_hbAlarm').value||0),
    };
    const roleLines = (el('s_teamRoles').value||'').split(/\r?\n/).map(s=>s.trim()).filter(Boolean);
    const newRoles = [];
    for (const line of roleLines) {
      const [k,l,srt] = line.split('|').map(x=>String(x||'').trim());
      if (!k || !l) continue;
      newRoles.push({key:k, label:l, sort: srt!=='' ? Number(srt) : 100});
    }
    const newTeam = {
      roles: newRoles.length ? newRoles : roles,
      allowMultipleLeads: el('s_multiLead').checked,
    };
    await api('/api/settings', {method:'POST', body:{key:'branding', value:newBrand}});
    await api('/api/settings', {method:'POST', body:{key:'privacy', value:newPrivacy}});
    await api('/api/settings', {method:'POST', body:{key:'ops', value:newOps}});
    await api('/api/settings', {method:'POST', body:{key:'team', value:newTeam}});
    toast('OK','Einstellungen gespeichert');
    const pub = await api('/api/public-settings');
    applyBrand(pub.settings);
  };

  const doUpload = async (key)=>{
    const inp = document.createElement('input');
    inp.type='file';
    inp.accept='.png,.jpg,.jpeg,.svg,.ico,.webp';
    inp.onchange = async ()=>{
      if (!inp.files?.[0]) return;
      const fd = new FormData();
      fd.append('file', inp.files[0]);
      fd.append('key', key);
      await api('/api/settings/upload', {method:'POST', body:fd, isForm:true});
      toast('OK','Upload gespeichert');
      await render();
    };
    inp.click();
  };
  el('btnLogo').onclick = ()=>doUpload('branding.logoPath');
  el('btnFav').onclick = ()=>doUpload('branding.faviconPath');
}

async function renderAdminTemplates(root){
  const rows = (await api('/api/settings')).settings;
  const map = Object.fromEntries(rows.map(r=>[r.key, r.value]));
  const templates = map.orderTemplates || [];

  root.innerHTML = `
    <div class="row" style="justify-content:space-between">
      <div class="small">Templates werden global gespeichert (Settings-Key: <code>orderTemplates</code>).</div>
      <button class="btn secondary" id="btnAddTpl">+ Template</button>
    </div>
    <hr />
    <div class="list" id="tplList"></div>
    <hr />
    <button class="btn" id="btnSaveTpl">Templates speichern</button>
  `;

  const list = root.querySelector('#tplList');
  const local = JSON.parse(JSON.stringify(templates));

  const renderList = ()=>{
    list.innerHTML='';
    local.forEach((t, idx)=>{
      const it = document.createElement('div');
      it.className='item';
      it.innerHTML = `
        <div class="row" style="justify-content:space-between;gap:10px">
          <div style="flex:1">
            <div class="row">
              <input data-k="key" data-idx="${idx}" placeholder="key" value="${escapeHtml(t.key||'')}" />
              <input data-k="name" data-idx="${idx}" placeholder="Name" style="flex:1" value="${escapeHtml(t.name||'')}" />
            </div>
            <div class="small" style="margin-top:8px">Felder (JSON):</div>
            <textarea data-k="fields" data-idx="${idx}" rows="5" style="width:100%">${escapeHtml(JSON.stringify(t.fields||[], null, 2))}</textarea>
          </div>
          <div class="row" style="flex-direction:column;align-items:flex-end">
            <button class="btn danger" data-del="${idx}">Löschen</button>
          </div>
        </div>
      `;
      it.querySelector('[data-del]').onclick = ()=>{ local.splice(idx,1); renderList(); };
      list.appendChild(it);
    });
  };

  renderList();

  root.querySelector('#btnAddTpl').onclick = ()=>{
    local.unshift({key:'', name:'', fields:[]});
    renderList();
  };

  root.querySelector('#btnSaveTpl').onclick = async ()=>{
    // pull inputs
    const inputs = root.querySelectorAll('input[data-idx], textarea[data-idx]');
    const next = JSON.parse(JSON.stringify(local));
    for (const inp of inputs) {
      const idx = Number(inp.getAttribute('data-idx'));
      const k = inp.getAttribute('data-k');
      if (!next[idx]) continue;
      if (k === 'fields') {
        try{
          next[idx].fields = JSON.parse(inp.value || '[]');
        } catch(e){
          return toast('Fehler', `Ungültiges JSON bei Template #${idx+1}`);
        }
      } else {
        next[idx][k] = inp.value.trim();
      }
    }
    // validate keys unique
    const keys = next.map(t=>t.key).filter(Boolean);
    const uniq = new Set(keys);
    if (uniq.size !== keys.length) return toast('Fehler','Template keys müssen eindeutig sein');
    await api('/api/settings', {method:'POST', body:{key:'orderTemplates', value:next}});
    toast('OK','Templates gespeichert');
  };
}

async function renderAdminOrgs(root){
  const res = await api('/api/admin/orgs');
  const orgs = res.orgs;
  root.innerHTML = `
    <div class="row" style="justify-content:space-between">
      <div class="small">Organisationen (Mandanten)</div>
      <button class="btn" id="btnOrgAdd">+ Organisation</button>
    </div>
    <hr />
    <div class="list" id="orgList"></div>
  `;
  const list = root.querySelector('#orgList');
  const renderRow = (o)=>{
    const it = document.createElement('div');
    it.className='item';
    it.innerHTML = `
      <div class="row" style="justify-content:space-between;gap:10px">
        <div style="flex:1">
          <div class="row">
            <input value="${escapeHtml(o.name||'')}" data-f="name" style="flex:2" />
            <input value="${escapeHtml(o.short_name||'')}" data-f="short_name" placeholder="Kurz" style="width:110px" />
            <input value="${escapeHtml(o.color||'')}" data-f="color" placeholder="#color" style="width:120px" />
          </div>
          <div class="small">ID: ${escapeHtml(o.id)}</div>
        </div>
        <div class="row" style="align-items:center">
          <label class="small"><input type="checkbox" data-f="is_active" ${o.is_active?'checked':''}/> aktiv</label>
          <button class="btn secondary" data-save>Speichern</button>
          <button class="btn danger" data-del ${o.is_active?'':'disabled'}>Deaktivieren</button>
        </div>
      </div>
    `;
    it.querySelector('[data-save]').onclick = async ()=>{
      const payload = {};
      for (const inp of it.querySelectorAll('[data-f]')) {
        const f = inp.getAttribute('data-f');
        payload[f] = inp.type==='checkbox' ? inp.checked : inp.value.trim();
      }
      await api(`/api/admin/orgs/${o.id}`, {method:'PUT', body:payload});
      toast('OK','Organisation gespeichert');
    };
    it.querySelector('[data-del]').onclick = async ()=>{
      if (!confirm('Organisation deaktivieren?')) return;
      await api(`/api/admin/orgs/${o.id}`, {method:'DELETE'});
      toast('OK','Deaktiviert');
      await render();
    };
    return it;
  };
  list.innerHTML='';
  for (const o of orgs) list.appendChild(renderRow(o));

  root.querySelector('#btnOrgAdd').onclick = async ()=>{
    const name = prompt('Name der Organisation?');
    if (!name) return;
    const short_name = prompt('Kurzname (optional)?') || '';
    const color = prompt('Farbe (optional, z.B. #d6002a)?') || '';
    await api('/api/admin/orgs', {method:'POST', body:{name, short_name, color}});
    toast('OK','Organisation erstellt');
    await render();
  };
}

async function renderAdminUsers(root){
  const [orgRes, userRes] = await Promise.all([
    api('/api/admin/orgs'),
    api('/api/admin/users'),
  ]);
  const orgs = orgRes.orgs;
  const users = userRes.users;
  const orgOpts = orgs.map(o=>`<option value="${escapeHtml(o.id)}">${escapeHtml(o.name)}</option>`).join('');

  root.innerHTML = `
    <div class="row" style="justify-content:space-between">
      <div class="small">Benutzer (Rollen: admin / el / gf / viewer)</div>
      <button class="btn" id="btnUserAdd">+ Benutzer</button>
    </div>
    <hr />
    <div class="list" id="userList"></div>
  `;
  const list = root.querySelector('#userList');
  list.innerHTML='';
  for (const u of users) {
    const it = document.createElement('div');
    it.className='item';
    it.innerHTML = `
      <div class="row" style="justify-content:space-between;gap:10px">
        <div style="flex:1">
          <div class="row">
            <div style="min-width:140px"><b>${escapeHtml(u.username)}</b></div>
            <input data-f="display_name" value="${escapeHtml(u.display_name||'')}" placeholder="Display" style="flex:1" />
            <select data-f="role">
              <option value="admin">admin</option>
              <option value="el">el</option>
              <option value="gf">gf</option>
              <option value="viewer">viewer</option>
            </select>
            <select data-f="org_id">${orgOpts}</select>
          </div>
          <div class="small">ID: ${escapeHtml(String(u.id))} · Org: ${escapeHtml(u.org_name)} · Active: ${u.is_active?'yes':'no'}</div>
        </div>
        <div class="row" style="align-items:center">
          <label class="small"><input type="checkbox" data-f="is_active" ${u.is_active?'checked':''}/> aktiv</label>
          <button class="btn secondary" data-save>Speichern</button>
          <button class="btn danger" data-reset>Passwort</button>
        </div>
      </div>
    `;
    it.querySelector('select[data-f="role"]').value = u.role;
    it.querySelector('select[data-f="org_id"]').value = u.org_id;
    it.querySelector('[data-save]').onclick = async ()=>{
      const payload = {};
      for (const inp of it.querySelectorAll('[data-f]')) {
        const f = inp.getAttribute('data-f');
        payload[f] = inp.type==='checkbox' ? inp.checked : inp.value;
      }
      await api(`/api/admin/users/${u.id}`, {method:'PUT', body:payload});
      toast('OK','User gespeichert');
    };
    it.querySelector('[data-reset]').onclick = async ()=>{
      const pw = prompt('Neues Passwort:');
      if (!pw) return;
      await api(`/api/admin/users/${u.id}/reset-password`, {method:'POST', body:{password:pw}});
      toast('OK','Passwort gesetzt');
    };
    list.appendChild(it);
  }

  root.querySelector('#btnUserAdd').onclick = async ()=>{
    const username = prompt('Username?');
    if (!username) return;
    const password = prompt('Start-Passwort?');
    if (!password) return;
    const display_name = prompt('Display Name (optional)?')||'';
    const role = prompt('Rolle (admin/el/gf/viewer)?','viewer')||'viewer';
    const org_id = orgs[0]?.id;
    await api('/api/admin/users', {method:'POST', body:{username, password, display_name, role, org_id}});
    toast('OK','Benutzer erstellt');
    await render();
  };
}

async function refreshAll(){
  await loadIncidents();
  await render();
}

function connectSSE(){
  if (state.sse) state.sse.close();
  const url = `/events?lastEventId=${encodeURIComponent(state.lastEventId)}`;
  const es = new EventSource(url, {withCredentials:true});
  state.sse = es;

  es.onmessage = (ev)=>{
    // default message event (unused)
  };
  es.addEventListener('hello', ()=>{});

  const handlers = {
    'incident.created': async ()=>{ await loadIncidents(); },
    'incident.updated': async ()=>{ await loadIncidents(); },
    'participants.updated': async (ev)=>{ if (state.page==='participants') await render(); },
    'teams.updated': async (ev)=>{ if (state.page==='teams') await render(); },
    'orders.updated': async (ev)=>{ if (state.page==='orders') await render(); },
    'chat.updated': async (ev)=>{ if (state.page==='chat') await render(); },
    'inbox.updated': async (ev)=>{ if (state.page==='inbox') await render(); toast('Inbox','Neues Angebot'); },
    'branding.updated': async ()=>{ const pub = await api('/api/public-settings'); applyBrand(pub.settings); },
    'settings.updated': async ()=>{ const pub = await api('/api/public-settings'); applyBrand(pub.settings); },
    'orgs.updated': async ()=>{ if (state.page==='admin') await render(); },
    'users.updated': async ()=>{ if (state.page==='admin') await render(); },
    'member.created': async ()=>{ if (state.page==='participants') await render(); },
    'capabilities.updated': async ()=>{ if (state.page==='participants') await render(); if (state.page==='admin') await render(); },
    'transfer.decided': async (ev)=>{ toast('Transfer', 'Entscheidung eingegangen'); },
  };

  for (const [type, fn] of Object.entries(handlers)) {
    es.addEventListener(type, async (ev)=>{
      if (ev.lastEventId) {
        state.lastEventId = ev.lastEventId;
        localStorage.setItem('lastEventId', state.lastEventId);
      }
      try{ await fn(ev); } catch(e){ console.warn(e); }
    });
  }

  es.onerror = ()=>{
    // auto-reconnect by browser; surface only if persistent
  };
}

// UI wiring
el('btnLogin').onclick = async ()=>{
  try{
    const username = el('loginUser').value.trim();
    const password = el('loginPass').value;
    const res = await api('/api/login', {method:'POST', body:{username, password}});
    state.user = res.user;
    state.csrf = res.user.csrf;
    state.settings = res.user.settings;
    applyBrand(state.settings);
    el('brandOrg').textContent = `${state.user.display_name} · ${state.user.role}`;
    setLoggedInUI(true);
    await loadIncidents();
    buildNav();
    connectSSE();
    render();
  } catch(e){ toast('Login fehlgeschlagen', e.message); }
};

el('btnLogout').onclick = async ()=>{
  try{ await api('/api/logout', {method:'POST'}); } catch(e){}
  state.user=null; state.csrf=null; state.incidentId=null;
  if (state.sse) state.sse.close();
  setLoggedInUI(false);
  toast('Logout','OK');
};

el('btnRefresh').onclick = refreshAll;
el('incidentSelect').onchange = async ()=>{
  state.incidentId = el('incidentSelect').value || null;
  localStorage.setItem('incidentId', state.incidentId||'');
  await render();
};

boot();
